<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Carrito;
use App\Models\Producto;
use App\Models\Venta;
use App\Models\DetalleVenta;
use App\Models\Orden;
use Illuminate\Support\Facades\Auth;

class CarritoController extends Controller
{
    public function index()
    {
        $carrito = Carrito::with('producto')->where('user_id', Auth::id())->get();
        return view('carrito.index', compact('carrito'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'producto_id' => 'required|exists:productos,id',
            'cantidad' => 'required|integer|min:1'
        ]);

        $producto = Producto::findOrFail($request->producto_id);

        $enCarrito = Carrito::where('user_id', Auth::id())
            ->where('producto_id', $producto->id)
            ->first();

        $cantidadSolicitada = $request->cantidad + ($enCarrito ? $enCarrito->cantidad : 0);

        if ($cantidadSolicitada > $producto->cantidad) {
            return redirect()->back()->with('error', 'No hay suficiente stock disponible para este producto.');
        }

        Carrito::updateOrCreate(
            [
                'user_id' => Auth::id(),
                'producto_id' => $producto->id,
            ],
            [
                'cantidad' => $cantidadSolicitada
            ]
        );

        return redirect()->back()->with('success', 'Producto agregado al carrito.');
    }

    public function update(Request $request, $id)
    {
        $carrito = Carrito::where('id', $id)->where('user_id', Auth::id())->firstOrFail();
        $producto = Producto::findOrFail($carrito->producto_id);

        $request->validate([
            'cantidad' => 'required|integer|min:0|max:' . $producto->cantidad
        ]);

        if ($request->cantidad == 0) {
            $carrito->delete();
            return redirect()->back()->with('success', 'Producto eliminado del carrito.');
        }

        $carrito->cantidad = $request->cantidad;
        $carrito->save();

        return redirect()->back()->with('success', 'Cantidad actualizada.');
    }

    public function destroy($id)
    {
        $carrito = Carrito::where('id', $id)->where('user_id', Auth::id())->firstOrFail();
        $carrito->delete();
        return redirect()->back()->with('success', 'Producto eliminado del carrito.');
    }

    public function confirmarVenta(Request $request)
    {
        $carrito = Carrito::with('producto')->where('user_id', Auth::id())->get();

        if ($carrito->isEmpty()) {
            return redirect()->route('carrito.index')->with('error', 'Tu carrito está vacío.');
        }

        foreach ($carrito as $item) {
            if ($item->cantidad > $item->producto->cantidad) {
                return redirect()->route('carrito.index')
                    ->with('error', 'No hay suficiente stock del producto: ' . $item->producto->nombre);
            }
        }

        $descuentos = [];
        $total = 0;

        foreach ($carrito as $item) {
            $subtotal = $item->producto->precio * $item->cantidad;

            $descuentoProducto = $request->input("descuento_producto.{$item->producto->id}", 0);
            $tipoDescuento = $request->input("tipo_descuento_producto.{$item->producto->id}", 'fijo');

            if ($tipoDescuento === 'porcentaje') {
                $descuentoAplicado = $subtotal * ($descuentoProducto / 100);
            } else {
                $descuentoAplicado = $descuentoProducto;
            }

            $totalProducto = max($subtotal - $descuentoAplicado, 0);
            $total += $totalProducto;

            $descuentos['productos'][$item->producto->id] = [
                'monto' => $descuentoAplicado,
                'tipo' => $tipoDescuento,
            ];
        }

        $descuentoGeneral = $request->input('descuento_general', 0);
        $tipoDescuentoGeneral = $request->input('tipo_descuento_general', 'fijo');

        if ($tipoDescuentoGeneral === 'porcentaje') {
            $descuentoGeneralAplicado = $total * ($descuentoGeneral / 100);
        } else {
            $descuentoGeneralAplicado = $descuentoGeneral;
        }

        $totalConDescuento = max($total - $descuentoGeneralAplicado, 0);

        $descuentos['general'] = [
            'monto' => $descuentoGeneralAplicado,
            'tipo' => $tipoDescuentoGeneral,
        ];

        // Cambios importantes para el flujo correcto:
        $metodoPagoSeleccionado = $request->input('metodo_pago', '');
        if (!$metodoPagoSeleccionado) {
            return redirect()->route('carrito.index')->with('error', 'Debe seleccionar un método de pago.');
        }
        $pagaCon = $request->input('paga_con', null);

        return view('ventas.confirmar', [
            'carrito' => $carrito,
            'total' => $total,
            'totalConDescuento' => $totalConDescuento,
            'descuentos' => $descuentos,
            'metodoPagoSeleccionado' => $metodoPagoSeleccionado,
            'pagaCon' => $pagaCon,
        ]);
    }

    public function procesarVenta(Request $request)
    {
        // Para depuración, descomenta la siguiente línea si necesitas ver el array recibido:
        dd($request->all());

        $carrito = Carrito::with('producto')->where('user_id', Auth::id())->get();

        if ($carrito->isEmpty()) {
            return redirect()->route('carrito.index')->with('error', 'Tu carrito está vacío.');
        }

        $descuentosProducto = $request->input('descuento_producto', []);
        $tiposDescuentoProducto = $request->input('tipo_descuento_producto', []);
        $descuentoGeneral = $request->input('descuento_general', 0);
        $tipoDescuentoGeneral = $request->input('tipo_descuento_general', 'fijo');

        $metodoPago = $request->input('metodo_pago', '');
        if (!$metodoPago) {
            return redirect()->route('carrito.index')->with('error', 'Método de pago no recibido.');
        }
        $pagaCon = $request->input('paga_con', null);
        $totalConDescuento = $request->input('total_con_descuento', 0);

        $descuentos = [];
        $total = 0;

        foreach ($carrito as $item) {
            $subtotal = $item->producto->precio * $item->cantidad;

            $prodId = $item->producto->id;
            $tipo = $tiposDescuentoProducto[$prodId] ?? 'fijo';
            $valorDescuento = $descuentosProducto[$prodId] ?? 0;

            if ($tipo === 'porcentaje') {
                $descuentoAplicado = $subtotal * ($valorDescuento / 100);
            } else {
                $descuentoAplicado = $valorDescuento;
            }

            $subtotalConDescuento = max($subtotal - $descuentoAplicado, 0);
            $total += $subtotalConDescuento;

            $descuentos['productos'][$prodId] = [
                'monto' => $descuentoAplicado,
                'tipo' => $tipo,
            ];
        }

        if ($tipoDescuentoGeneral === 'porcentaje') {
            $descuentoGeneralAplicado = $total * ($descuentoGeneral / 100);
        } else {
            $descuentoGeneralAplicado = $descuentoGeneral;
        }

        $totalConDescuento = max($total - $descuentoGeneralAplicado, 0);

        $descuentos['general'] = [
            'monto' => $descuentoGeneralAplicado,
            'tipo' => $tipoDescuentoGeneral,
        ];

        $venta = Venta::create([
            'user_id' => Auth::id(),
            'fecha' => now(),
            'total' => $totalConDescuento,
            'descuentos' => json_encode($descuentos),
            'metodo_pago' => $metodoPago,
            'paga_con' => $pagaCon,
        ]);

        $vuelto = null;
        if ($metodoPago === 'Cash' && is_numeric($pagaCon)) {
            $vuelto = $pagaCon - $totalConDescuento;
        }

        foreach ($carrito as $item) {
            $producto = $item->producto;
            $producto->cantidad -= $item->cantidad;
            $producto->save();

            DetalleVenta::create([
                'venta_id' => $venta->id,
                'producto_id' => $producto->id,
                'cantidad' => $item->cantidad,
                'precio' => $producto->precio,
                'subtotal' => $producto->precio * $item->cantidad,
                'descuento' => $descuentos['productos'][$producto->id]['monto'] ?? 0,
                'total_final' => $totalConDescuento,
                'metodo_pago' => $metodoPago,
                'paga_con' => $pagaCon,
                'vuelto' => $vuelto,
            ]);
        }

        Carrito::where('user_id', Auth::id())->delete();

        return redirect()->route('ventas.show', $venta->id)
            ->with('success', 'Venta procesada correctamente');
    }
}