@extends('layouts.app')

@section('title', 'Historial de Ventas')

@section('content')
    <div class="container-fluid">
        <div class="my-4">
            <h1>Historial de Ventas</h1>
        </div>

        {{-- Cuadro para mostrar el total de ventas (solo si hay una búsqueda) --}}
        @if(request('busqueda') || request('fecha_inicio') || request('fecha_fin'))
            <div class="row mb-4">
                <div class="col-md-4 mx-auto">
                    <div class="card text-center bg-primary text-white shadow-lg">
                        <div class="card-header">
                            Total de Ventas
                        </div>
                        <div class="card-body">
                            <h3 class="card-title">${{ number_format($totalVentas, 2) }}</h3>
                        </div>
                    </div>
                </div>
            </div>
        @endif

        {{-- Buscador combinado: texto y fechas --}}
        <form method="GET" action="{{ route('ventas.index') }}" class="row g-3 align-items-end mb-4">
            <div class="col-md-4 col-lg-auto">
                <label for="busqueda" class="form-label mb-0">Buscar</label>
                <input type="text" name="busqueda" id="busqueda" class="form-control"
                        value="{{ request('busqueda') }}" placeholder="ID Venta, Vendedor...">
            </div>
            <div class="col-md-4 col-lg-auto">
                <label for="fecha_inicio" class="form-label mb-0">Fecha inicio</label>
                <input type="date" name="fecha_inicio" id="fecha_inicio" class="form-control"
                        value="{{ request('fecha_inicio', '') }}">
            </div>
            <div class="col-md-4 col-lg-auto">
                <label for="fecha_fin" class="form-label mb-0">Fecha fin</label>
                <input type="date" name="fecha_fin" id="fecha_fin" class="form-control"
                        value="{{ request('fecha_fin', '') }}">
            </div>
            <div class="col-md-12 col-lg-auto d-flex justify-content-end">
                <button type="submit" class="btn btn-primary me-2">
                    <i class="bi bi-search"></i>
                    Buscar
                </button>
                <a href="{{ route('ventas.index') }}" class="btn btn-outline-secondary">
                    <i class="bi bi-x-circle"></i>
                    Limpiar
                </a>
            </div>
        </form>

        {{-- Botones de exportación --}}
        <div class="d-flex mb-3">
            <a href="{{ route('ventas.exportarPdf', request()->all()) }}" class="btn btn-success me-2" target="_blank">
                <i class="bi bi-file-earmark-pdf"></i> Exportar a PDF
            </a>
            <a href="{{ route('ventas.exportarExcel', request()->all()) }}" class="btn btn-success" target="_blank">
                <i class="bi bi-file-earmark-excel"></i> Exportar a Excel
            </a>
        </div>

        {{-- Mensajes de éxito o error --}}
        @if (session('success'))
            <div class="alert alert-success">{{ session('success') }}</div>
        @endif
        @if (session('error'))
            <div class="alert alert-danger">{{ session('error') }}</div>
        @endif

        <div class="table-responsive">
            <table class="table table-bordered table-hover">
                <thead>
                    <tr>
                        <th>ID Venta</th>
                        <th>Vendedor</th>
                        <th>Fecha</th>
                        <th>Total</th>
                        <th>Acciones</th>
                    </tr>
                </thead>
                <tbody>
                    @forelse($ventas as $venta)
                        <tr>
                            <td>{{ $venta->id }}</td>
                            <td>{{ $venta->user->name ?? '-' }}</td>
                            <td>{{ $venta->created_at->format('d/m/Y H:i') }}</td>
                            <td>${{ number_format($venta->total, 2) }}</td>
                            <td class="d-flex justify-content-center align-items-center">
                                <a href="{{ route('ventas.show', $venta->id) }}" class="btn btn-sm btn-info me-2">
                                    <i class="bi bi-eye"></i> Detalles
                                </a>
                                <!-- Botones para eliminar y editar la venta, solo si el usuario es un administrador -->
                                @if(Auth::check() && Auth::user()->is_admin)
                                <button type="button" class="btn btn-sm btn-warning me-2" data-bs-toggle="modal" data-bs-target="#editDateModal" data-id="{{ $venta->id }}" data-date="{{ $venta->created_at->format('Y-m-d\TH:i') }}">
                                    <i class="bi bi-pencil"></i> Editar
                                </button>
                                <form action="{{ route('ventas.destroy', $venta->id) }}" method="POST"
                                    onsubmit="return confirm('¿Estás seguro de que quieres eliminar esta venta? Esta acción no se puede deshacer y el stock del producto NO se devolverá al inventario.');">
                                    @csrf
                                    @method('DELETE')
                                    <button type="submit" class="btn btn-sm btn-danger">
                                        <i class="bi bi-trash"></i> Eliminar
                                    </button>
                                </form>
                                @endif
                            </td>
                        </tr>
                    @empty
                        <tr>
                            <td colspan="5">No hay ventas registradas.</td>
                        </tr>
                    @endforelse
                </tbody>
            </table>
        </div>
        
        <div class="mb-2">
            Mostrando {{ $ventas->firstItem() }} a {{ $ventas->lastItem() }} de {{ $ventas->total() }} resultados
        </div>
        <div class="d-flex justify-content-center mt-4">
            {{ $ventas->appends(request()->query())->links('vendor.pagination.bootstrap-5') }}
        </div>

        <!-- Modal para editar la fecha de venta -->
        <div class="modal fade" id="editDateModal" tabindex="-1" aria-labelledby="editDateModalLabel" aria-hidden="true">
          <div class="modal-dialog">
            <div class="modal-content">
              <div class="modal-header">
                <h5 class="modal-title" id="editDateModalLabel">Editar Fecha de Venta</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
              </div>
              <form id="editDateForm" action="{{ route('ventas.update-date') }}" method="POST">
                @csrf
                @method('PUT')
                <div class="modal-body">
                  <input type="hidden" name="venta_id" id="ventaIdInput">
                  <div class="mb-3">
                    <label for="newDateInput" class="form-label">Nueva Fecha y Hora</label>
                    <input type="datetime-local" class="form-control" id="newDateInput" name="new_date" required>
                  </div>
                </div>
                <div class="modal-footer">
                  <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                  <button type="submit" class="btn btn-primary">Guardar Cambios</button>
                </div>
              </form>
            </div>
          </div>
        </div>
    </div>
    
    {{-- Agregamos la librería de Bootstrap y el script al final para garantizar su funcionamiento --}}
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            var editDateModal = document.getElementById('editDateModal');
            if (editDateModal) {
                editDateModal.addEventListener('show.bs.modal', function(event) {
                    var button = event.relatedTarget;
                    var ventaId = button.getAttribute('data-id');
                    var ventaDate = button.getAttribute('data-date');
                    
                    var modalForm = document.getElementById('editDateForm');
                    var ventaIdInput = modalForm.querySelector('#ventaIdInput');
                    var newDateInput = modalForm.querySelector('#newDateInput');

                    if (ventaIdInput && newDateInput) {
                        ventaIdInput.value = ventaId;
                        newDateInput.value = ventaDate;
                    }
                });
            }
        });
    </script>
@endsection
