@extends('layouts.app')

@section('content')
<div class="container">
    <h1>Órdenes generadas por stock mínimo</h1>

    @if(session('success'))
        <div class="alert alert-success">{{ session('success') }}</div>
    @endif

    {{-- Buscador y botones de acción --}}
    <div class="d-flex justify-content-between align-items-end mb-4">
        <form method="GET" action="{{ route('ordenes.index') }}" class="row g-3 align-items-end flex-grow-1 me-3" id="search-form">
            <div class="col-auto">
                <label for="search" class="form-label mb-0">Buscar Producto</label>
                <input type="text" name="search" id="search" class="form-control"
                       value="{{ request('search') }}" placeholder="Nombre, código...">
            </div>
            <div class="col-auto">
                <button type="submit" class="btn btn-primary">
                    <i class="bi bi-search"></i> Buscar
                </button>
                <a href="#" id="clear-search-btn" class="btn btn-outline-secondary">
                    <i class="bi bi-x-circle"></i> Limpiar
                </a>
            </div>
        </form>
        <div class="col-auto">
            <a href="#" id="clear-quantities-btn" class="btn btn-danger">
                <i class="bi bi-eraser"></i> Limpiar Órdenes
            </a>
        </div>
    </div>

    {{-- Botones de exportar --}}
    <div class="mb-3">
        <a href="{{ route('ordenes.exportar-pdf') }}" class="btn btn-success me-2" id="export-pdf-btn">
            <i class="bi bi-file-earmark-pdf"></i> Exportar a PDF
        </a>
        <a href="{{ route('ordenes.exportar-excel') }}" class="btn btn-success" id="export-excel-btn">
            <i class="bi bi-file-earmark-excel"></i> Exportar a Excel
        </a>
    </div>

    <form action="{{ route('ordenes.actualizar') }}" method="POST" id="quantities-form">
        @csrf
        <table class="table table-bordered">
            <thead>
                <tr>
                    <th>Producto</th>
                    <th>Código</th>
                    <th>Marca</th>
                    <th>Cantidad actual</th>
                    <th>Stock mínimo</th>
                    <th>Stock máximo</th>
                    <th>Cantidad a pedir</th>
                    <th>Precio</th>
                    <th>Costo</th>
                    <th>Cantidad a enviar</th>
                </tr>
            </thead>
            <tbody>
                @forelse($ordenes as $orden)
                    <tr>
                        <td>{{ $orden->nombre }}</td>
                        <td>{{ $orden->codigo }}</td>
                        <td>{{ $orden->producto->marca ?? '-' }}</td>
                        <td>
                            @if($orden->producto)
                                @php
                                    $cantidad_actual = $orden->producto->cantidad;
                                    $stock_minimo = $orden->producto->minimo;
                                    $color_class = ($cantidad_actual <= $stock_minimo) ? 'bg-danger' : 'bg-primary';
                                @endphp
                                <span class="badge {{ $color_class }}">
                                    {{ $cantidad_actual }}
                                </span>
                            @else
                                <span class="badge bg-secondary">Sin producto</span>
                            @endif
                        </td>
                        <td>{{ $orden->minimo }}</td>
                        <td>{{ $orden->maximo }}</td>
                        <td>
                            @if($orden->producto)
                                @php
                                    $cantidad_a_pedir = max(0, $orden->maximo - $orden->producto->cantidad);
                                @endphp
                                {{ $cantidad_a_pedir }}
                            @else
                                <span class="text-muted">N/A</span>
                            @endif
                        </td>
                        <td>${{ number_format($orden->precio, 2) }}</td>
                        <td>${{ number_format($orden->producto->costo ?? 0, 2) }}</td>
                        <td>
                            <input type="number" class="form-control quantities-input" name="cantidades[{{ $orden->id }}]" min="0" placeholder="Cantidad">
                        </td>
                    </tr>
                @empty
                    <tr>
                        <td colspan="9">No hay órdenes generadas.</td>
                    </tr>
                @endforelse
            </tbody>
        </table>
        @if($ordenes->count() > 0)
            <div class="text-end">
                <button class="btn btn-success" type="submit">Orden actualizada</button>
            </div>
        @endif
    </form>

    {{-- Contenedor para centrar la paginación --}}
    <div class="d-flex justify-content-center">
        {{ $ordenes->appends(request()->input())->links() }}
    </div>
</div>

<script>
    document.addEventListener('DOMContentLoaded', function() {
        const quantitiesForm = document.getElementById('quantities-form');
        const quantitiesInputs = document.querySelectorAll('.quantities-input');
        const clearSearchBtn = document.getElementById('clear-search-btn');
        const clearQuantitiesBtn = document.getElementById('clear-quantities-btn');
        const exportPdfBtn = document.getElementById('export-pdf-btn');
        const exportExcelBtn = document.getElementById('export-excel-btn');

        // 1. Cargamos las cantidades guardadas en localStorage al iniciar
        let storedQuantities = JSON.parse(localStorage.getItem('ordenesQuantities')) || {};

        // 2. Rellenamos los campos de la página actual con los valores guardados
        quantitiesInputs.forEach(input => {
            const productId = input.name.match(/\[(\d+)\]/)[1];
            if (storedQuantities[productId] !== undefined) {
                input.value = storedQuantities[productId];
            }
        });

        // 3. Guardamos los cambios en localStorage cada vez que se modifica un campo
        quantitiesInputs.forEach(input => {
            input.addEventListener('input', function() {
                const productId = this.name.match(/\[(\d+)\]/)[1];
                if (this.value > 0) {
                    storedQuantities[productId] = this.value;
                } else {
                    delete storedQuantities[productId];
                }
                localStorage.setItem('ordenesQuantities', JSON.stringify(storedQuantities));
                updateExportLinks(); // También actualizamos los enlaces al cambiar una cantidad
            });
        });

        // 4. Manejador para el envío del formulario
        quantitiesForm.addEventListener('submit', function(e) {
            // No prevenimos el envío, ya que el formulario no contiene todos los inputs.
            // En su lugar, añadimos los campos ocultos antes de enviar.
            const allQuantities = JSON.parse(localStorage.getItem('ordenesQuantities')) || {};
            for (const productId in allQuantities) {
                const hiddenInput = document.createElement('input');
                hiddenInput.type = 'hidden';
                hiddenInput.name = `cantidades[${productId}]`;
                hiddenInput.value = allQuantities[productId];
                this.appendChild(hiddenInput);
            }
            // Ahora el formulario se enviará con todos los datos
        });

        // 5. Función para agregar los parámetros de cantidad a los enlaces de exportación
        function updateExportLinks() {
            const allQuantities = JSON.parse(localStorage.getItem('ordenesQuantities')) || {};
            const searchParam = document.getElementById('search').value;
            const exportParams = new URLSearchParams();
            if (searchParam) {
                exportParams.set('search', searchParam);
            }
            for (const productId in allQuantities) {
                exportParams.set(`cantidades[${productId}]`, allQuantities[productId]);
            }
            const pdfUrl = `${exportPdfBtn.href.split('?')[0]}?${exportParams.toString()}`;
            const excelUrl = `${exportExcelBtn.href.split('?')[0]}?${exportParams.toString()}`;

            exportPdfBtn.href = pdfUrl;
            exportExcelBtn.href = excelUrl;
        }

        // 6. Manejador del botón "Limpiar buscador"
        clearSearchBtn.addEventListener('click', function(e) {
            e.preventDefault();
            const url = new URL(window.location.href);
            const params = new URLSearchParams(url.search);
            params.delete('search');
            window.location.href = `${url.origin}${url.pathname}?${params.toString()}`;
        });

        // 7. Manejador del botón "Limpiar Órdenes"
        clearQuantitiesBtn.addEventListener('click', function(e) {
            e.preventDefault();
            localStorage.removeItem('ordenesQuantities');
            window.location.href = window.location.pathname; // Redirige a la URL base
        });

        // 8. Limpiamos localStorage después de una actualización exitosa
        @if(session('success'))
            localStorage.removeItem('ordenesQuantities');
        @endif
    });
</script>
@endsection
