<?php

namespace App\Http\Controllers;

use App\Models\Orden;
use App\Models\Producto;
use Illuminate\Http\Request;
use Barryvdh\DomPDF\Facade\Pdf;
use App\Exports\OrdenesExport;
use Maatwebsite\Excel\Facades\Excel;

class OrdenController extends Controller
{
    /**
     * Muestra la lista de órdenes automáticas generadas por stock mínimo.
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\View\View
     */
    public function index(Request $request)
    {
        $query = Orden::with('producto');

        // Lógica de búsqueda
        if ($request->has('search') && $request->search != '') {
            $search = $request->search;
            $query->whereHas('producto', function ($q) use ($search) {
                $q->where('nombre', 'like', "%{$search}%")
                  ->orWhere('codigo', 'like', "%{$search}%");
            });
        }

        // Aumentamos el número de órdenes por página a 20
        $ordenes = $query->paginate(20);

        return view('ordenes.index', compact('ordenes'));
    }

    /**
     * Procesa la actualización masiva de órdenes.
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function actualizar(Request $request)
    {
        $cantidades = $request->input('cantidades', []);

        $eliminadas = 0;
        $actualizadas = 0;

        foreach ($cantidades as $ordenId => $cantidadEnviar) {
            $orden = Orden::find($ordenId);

            if (!$orden) continue;

            // Obtener el producto relacionado correctamente
            $producto = $orden->producto ?? Producto::find($orden->producto_id);
            if (!$producto) continue;

            $cantidadEnviar = intval($cantidadEnviar);
            if ($cantidadEnviar < 1) continue;

            // Actualiza la cantidad actual del producto
            $producto->cantidad += $cantidadEnviar;
            $producto->save();

            // Determina el stock máximo correctamente (tu tabla es 'maximo', no 'stock_maximo')
            $stockMaximo = $producto->maximo;

            if ($stockMaximo !== null && $producto->cantidad >= $stockMaximo) {
                // Si alcanza o supera el stock máximo, elimina la orden
                $orden->delete();
                $eliminadas++;
            } else {
                // Si no, la orden sigue activa
                $actualizadas++;
            }
        }

        $msg = [];
        if ($eliminadas) $msg[] = "$eliminadas orden(es) completadas y eliminadas";
        if ($actualizadas) $msg[] = "$actualizadas orden(es) actualizadas";
        if (!$eliminadas && !$actualizadas) $msg[] = "No se realizaron cambios";

        return redirect()->route('ordenes.index')->with('success', implode('. ', $msg) . '.');
    }

    /**
     * Genera y exporta un PDF de las órdenes generadas.
     * @return \Illuminate\Http\Response
     */
    public function exportarPDF(Request $request)
    {
        $ordenes = Orden::with('producto')->get();
        $pdf = Pdf::loadView('ordenes.pdf', compact('ordenes'));
        return $pdf->download('ordenes_stock_minimo.pdf');
    }

    /**
     * Genera y exporta un archivo de Excel de las órdenes.
     * @return \Illuminate\Http\Response
     */
    public function exportarExcel(Request $request)
    {
        // Obtener los parámetros de búsqueda y las cantidades desde la URL
        $search = $request->input('search', '');
        $cantidades = $request->input('cantidades', []);
        
        // Se crea una instancia de OrdenesExport y se le pasan los datos necesarios
        return Excel::download(new OrdenesExport($search, $cantidades), 'ordenes_stock_minimo.xlsx');
    }

    /**
     * Elimina una orden (cuando el admin presiona "orden actualizada" individual, ya no se usa).
     * @param \App\Models\Orden $orden
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy(Orden $orden)
    {
        $orden->delete();
        return redirect()->route('ordenes.index')->with('success', 'Orden eliminada correctamente.');
    }
}
