<?php

namespace App\Exports;

use App\Models\Orden;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;
use Illuminate\Support\Collection;

class OrdenesExport implements FromCollection, WithHeadings, WithMapping
{
    // Añadimos propiedades para almacenar los datos que se pasan al constructor
    protected $search;
    protected $cantidades;

    // Se necesita un constructor para recibir los datos del controlador
    public function __construct($search, $cantidades)
    {
        $this->search = $search;
        $this->cantidades = $cantidades;
    }

    /**
    * @return \Illuminate\Support\Collection
    */
    public function collection()
    {
        // 1. Obtener todas las órdenes con el producto relacionado
        $query = Orden::with('producto');

        // 2. Aplicar el filtro de búsqueda si existe
        if ($this->search) {
            $search = $this->search;
            $query->whereHas('producto', function ($q) use ($search) {
                $q->where('nombre', 'like', "%{$search}%")
                  ->orWhere('codigo', 'like', "%{$search}%");
            });
        }
        
        // Retorna la colección de órdenes filtrada
        return $query->get();
    }

    /**
     * @return array
     */
    public function headings(): array
    {
        // Se añaden los nuevos encabezados para que coincidan con los datos
        return [
            'Producto',
            'Código',
            'Marca',
            'Cantidad actual',
            'Stock mínimo',
            'Stock máximo',
            'Cantidad a pedir',
            'Precio',
            'Costo',
            'Cantidad a enviar',
        ];
    }

    /**
     * @param mixed $orden
     * @return array
     */
    public function map($orden): array
    {
        $cantidad_actual = $orden->producto->cantidad ?? 0;
        $cantidad_a_pedir = max(0, $orden->maximo - $cantidad_actual);
        // Se obtiene la cantidad a enviar del array recibido en el constructor
        $cantidad_a_enviar = $this->cantidades[$orden->id] ?? null;

        // Se retorna un array con todos los campos en el orden correcto
        return [
            $orden->nombre,
            $orden->codigo,
            $orden->producto->marca ?? '-',
            $cantidad_actual,
            $orden->minimo,
            $orden->maximo,
            $cantidad_a_pedir,
            number_format($orden->precio, 2),
            number_format($orden->producto->costo ?? 0, 2),
            $cantidad_a_enviar,
        ];
    }
}
