<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Venta;
use App\Models\Producto;
use App\Models\Carrito;
use App\Models\DetalleVenta;
use App\Models\Orden;
use Illuminate\Support\Facades\Auth;
use Barryvdh\DomPDF\Facade\Pdf;
use App\Exports\VentasExport;
use Maatwebsite\Excel\Facades\Excel;

class VentaController extends Controller
{
    public function index(Request $request)
    {
        $query = Venta::with(['user', 'detalles.producto'])->orderBy('created_at', 'desc');

        if ($request->filled('busqueda')) {
            $busqueda = $request->input('busqueda');
            $query->where(function ($q) use ($busqueda) {
                $q->where('id', $busqueda)
                    ->orWhereHas('user', function ($q2) use ($busqueda) {
                        $q2->where('name', 'like', '%' . $busqueda . '%');
                    });
            });
        }

        if ($request->filled('fecha_inicio')) {
            $query->whereDate('created_at', '>=', $request->input('fecha_inicio'));
        }

        if ($request->filled('fecha_fin')) {
            $query->whereDate('created_at', '<=', $request->input('fecha_fin'));
        }

        $ventas = $query->paginate(10)->appends($request->all());

        $fecha_inicio = $request->input('fecha_inicio', '');
        $fecha_fin = $request->input('fecha_fin', '');

        return view('ventas.index', compact('ventas', 'fecha_inicio', 'fecha_fin'));
    }

    public function show(Venta $venta)
    {
        $venta->load(['user', 'detalles.producto']);
        $descuentos = json_decode($venta->descuentos, true);

        return view('ventas.show', compact('venta', 'descuentos'));
    }

    public function confirmar(Request $request)
    {
        $userId = Auth::id();
        $carrito = Carrito::with('producto')->where('user_id', $userId)->get();

        if ($carrito->isEmpty()) {
            return redirect()->route('carrito.index')->with('error', 'El carrito está vacío.');
        }

        foreach ($carrito as $item) {
            if ($item->cantidad > $item->producto->cantidad) {
                return redirect()->route('carrito.index')
                    ->with('error', 'No hay suficiente stock para: ' . $item->producto->nombre);
            }
        }

        $descuentosProducto = $request->input('descuento_producto', []);
        $tiposDescuentoProducto = $request->input('tipo_descuento_producto', []);
        $descuentoGeneral = $request->input('descuento_general', 0);
        $tipoDescuentoGeneral = $request->input('tipo_descuento_general', 'fijo');

        $descuentosProductoPorId = [];
        $tiposDescuentoProductoPorId = [];

        foreach ($carrito as $item) {
            $prodId = $item->producto->id;
            $descuentosProductoPorId[$prodId] = $descuentosProducto[$prodId] ?? 0;
            $tiposDescuentoProductoPorId[$prodId] = $tiposDescuentoProducto[$prodId] ?? 'fijo';
        }

        $totales = $this->calcularTotalesConDescuento(
            $carrito,
            $descuentosProductoPorId,
            $tiposDescuentoProductoPorId,
            $descuentoGeneral,
            $tipoDescuentoGeneral
        );

        $total = $totales['total'];
        $totalConDescuento = $totales['totalConDescuento'];

        $descuentos = [
            'productos' => $totales['descuentosProducto'],
            'general' => [
                'monto' => $totales['descuentoGeneral'],
                'tipo' => $totales['tipoDescuentoGeneral'],
            ],
        ];

        return view('ventas.confirmar', compact('carrito', 'total', 'totalConDescuento', 'descuentos'));
    }

    public function procesar(Request $request)
    {
        //dd($request->all());
        $userId = Auth::id();
        $carrito = Carrito::with('producto')->where('user_id', $userId)->get();

        if ($carrito->isEmpty()) {
            return redirect()->route('carrito.index')->with('error', 'El carrito está vacío.');
        }

        foreach ($carrito as $item) {
            if ($item->cantidad > $item->producto->cantidad) {
                return redirect()->route('carrito.index')
                    ->with('error', 'No hay suficiente stock para: ' . $item->producto->nombre);
            }
        }

        // Recoger descuentos individuales
        $descuentosProducto = [];
        $tiposDescuentoProducto = [];

        foreach ($carrito as $item) {
            $prodId = $item->producto->id;
            $descuento = floatval($request->input("descuento_producto.$prodId", 0));
            $tipo = $request->input("tipo_descuento_producto.$prodId", 'fijo');

            $subtotal = $item->producto->precio * $item->cantidad;
            if ($tipo === 'porcentaje' && $descuento > 100) {
                $descuento = 100;
            } elseif ($tipo === 'fijo' && $descuento > $subtotal) {
                $descuento = $subtotal;
            }

            $descuentosProducto[$prodId] = $descuento;
            $tiposDescuentoProducto[$prodId] = $tipo;
        }

        $descuentoGeneral = floatval($request->input('descuento_general', 0));
        $tipoDescuentoGeneral = $request->input('tipo_descuento_general', 'fijo');
        $totalFinalConfirmado = floatval($request->input('total_con_descuento', 0));

        // Calcular totales y descuentos
        $totales = $this->calcularTotalesConDescuento(
            $carrito,
            $descuentosProducto,
            $tiposDescuentoProducto,
            $descuentoGeneral,
            $tipoDescuentoGeneral
        );

        $venta = Venta::create([
            'user_id' => $userId,
            'fecha' => now(),
            'total' => $totalFinalConfirmado,
            'descuentos' => json_encode([
                'productos' => $totales['descuentosProducto'],
                'general' => [
                    'monto' => $totales['descuentoGeneral'],
                    'tipo' => $tipoDescuentoGeneral
                ]
            ])
        ]);

        foreach ($carrito as $item) {
            $producto = $item->producto;
            $prodId = $producto->id;

            $subtotal = $producto->precio * $item->cantidad;
            $descuento = $totales['descuentosProducto'][$prodId]['monto'] ?? 0;

            $producto->cantidad -= $item->cantidad;
            $producto->save();

            DetalleVenta::create([
                'venta_id' => $venta->id,
                'producto_id' => $prodId,
                'cantidad' => $item->cantidad,
                'precio' => $producto->precio,
                'subtotal' => $subtotal,
                'descuento' => $descuento,
                'total_final' => $totalFinalConfirmado,
                'metodo_pago' => $request->input('metodo_pago'), // <-- ¡AQUÍ!
            ]);

            $this->verificarOrdenReposicion($producto);
        }

        Carrito::where('user_id', $userId)->delete();

        return redirect()->route('ventas.show', $venta->id)
            ->with('success', 'Venta procesada correctamente');
    }





    private function calcularTotalesConDescuento($carrito, $descuentosProducto, $tiposDescuentoProducto, $descuentoGeneral, $tipoDescuentoGeneral)
    {
        $total = 0;
        $descuentos = [];

        foreach ($carrito as $item) {
            $prodId = $item->producto->id;
            $subtotal = $item->producto->precio * $item->cantidad;
            $descuento = $descuentosProducto[$prodId] ?? 0;
            $tipo = $tiposDescuentoProducto[$prodId] ?? 'fijo';

            $descuentoAplicado = $tipo === 'porcentaje'
                ? $subtotal * ($descuento / 100)
                : $descuento;

            $totalProducto = max($subtotal - $descuentoAplicado, 0);
            $total += $totalProducto;

            $descuentos[$prodId] = [
                'monto' => $descuentoAplicado,
                'tipo' => $tipo,
                'subtotal' => $subtotal,
                'total' => $totalProducto,
            ];
        }

        $descuentoGeneralAplicado = $tipoDescuentoGeneral === 'porcentaje'
            ? $total * ($descuentoGeneral / 100)
            : $descuentoGeneral;

        $totalConDescuento = max($total - $descuentoGeneralAplicado, 0);

        return [
            'total' => $total,
            'totalConDescuento' => $totalConDescuento,
            'descuentoGeneral' => $descuentoGeneralAplicado,
            'descuentosProducto' => $descuentos,
            'tipoDescuentoGeneral' => $tipoDescuentoGeneral
        ];
    }

    private function verificarOrdenReposicion($producto)
    {
        if ($producto->cantidad <= $producto->minimo) {
            $cantidadOrden = $producto->maximo - $producto->cantidad;

            Orden::firstOrCreate(
                ['producto_id' => $producto->id, 'estado' => 'pendiente'],
                [
                    'nombre' => $producto->nombre,
                    'codigo' => $producto->codigo,
                    'precio' => $producto->precio,
                    'cantidad' => $cantidadOrden,
                    'minimo' => $producto->minimo,
                    'maximo' => $producto->maximo,
                    'descripcion' => $producto->descripcion,
                    'imagen' => $producto->imagen,
                    'estado' => 'pendiente'
                ]
            );
        }
    }
    public function exportarPdf(Request $request)
    {
        $query = Venta::with(['user', 'detalles.producto'])->orderBy('created_at', 'desc');

        if ($request->filled('busqueda')) {
            $busqueda = $request->input('busqueda');
            $query->where(function ($q) use ($busqueda) {
                $q->where('id', $busqueda)
                    ->orWhereHas('user', function ($q2) use ($busqueda) {
                        $q2->where('name', 'like', '%' . $busqueda . '%');
                    });
            });
        }

        if ($request->filled('fecha_inicio')) {
            $query->whereDate('created_at', '>=', $request->input('fecha_inicio'));
        }

        if ($request->filled('fecha_fin')) {
            $query->whereDate('created_at', '<=', $request->input('fecha_fin'));
        }

        $ventas = $query->get();

        // Puedes pasar también los filtros si los necesitas mostrar en el PDF
        $fecha_inicio = $request->input('fecha_inicio', '');
        $fecha_fin = $request->input('fecha_fin', '');

        $pdf = Pdf::loadView('ventas.pdf', compact('ventas', 'fecha_inicio', 'fecha_fin'))
            ->setPaper('a4', 'portrait'); // o 'landscape' si quieres horizontal

        return $pdf->download('historial_ventas.pdf');
    }
    public function exportarExcel(Request $request)
    {
        $busqueda = $request->input('busqueda');
        $fecha_inicio = $request->input('fecha_inicio');
        $fecha_fin = $request->input('fecha_fin');

        return Excel::download(
            new VentasExport($busqueda, $fecha_inicio, $fecha_fin),
            'historial_ventas.xlsx'
        );
    }
}