<?php

namespace App\Http\Controllers;

use App\Models\Orden;
use App\Models\Producto;
use Illuminate\Http\Request;

class OrdenController extends Controller
{
    /**
     * Muestra la lista de órdenes automáticas generadas por stock mínimo.
     */
    public function index()
    {
        $ordenes = Orden::with('producto')->paginate(10);
        return view('ordenes.index', compact('ordenes'));
    }

    /**
     * Procesa la actualización masiva de órdenes.
     */
    public function actualizar(Request $request)
    {
        $cantidades = $request->input('cantidades', []);

        $eliminadas = 0;
        $actualizadas = 0;

        foreach ($cantidades as $ordenId => $cantidadEnviar) {
            $orden = Orden::find($ordenId);

            if (!$orden) continue;

            // Obtener el producto relacionado correctamente
            // Si tienes la relación producto en el modelo Orden, usa $orden->producto
            // Si no, usa Producto::find($orden->producto_id)
            $producto = $orden->producto ?? Producto::find($orden->producto_id);
            if (!$producto) continue;

            $cantidadEnviar = intval($cantidadEnviar);
            if ($cantidadEnviar < 1) continue;

            // Actualiza la cantidad actual del producto
            $producto->cantidad += $cantidadEnviar;
            $producto->save();

            // Determina el stock máximo correctamente (tu tabla es 'maximo', no 'stock_maximo')
            $stockMaximo = $producto->maximo;

            if ($stockMaximo !== null && $producto->cantidad >= $stockMaximo) {
                // Si alcanza o supera el stock máximo, elimina la orden
                $orden->delete();
                $eliminadas++;
            } else {
                // Si no, la orden sigue activa
                $actualizadas++;
            }
        }

        $msg = [];
        if ($eliminadas) $msg[] = "$eliminadas orden(es) completadas y eliminadas";
        if ($actualizadas) $msg[] = "$actualizadas orden(es) actualizadas";
        if (!$eliminadas && !$actualizadas) $msg[] = "No se realizaron cambios";

        return redirect()->route('ordenes.index')->with('success', implode('. ', $msg) . '.');
    }

    /**
     * Elimina una orden (cuando el admin presiona "orden actualizada" individual, ya no se usa).
     */
    public function destroy(Orden $orden)
    {
        $orden->delete();
        return redirect()->route('ordenes.index')->with('success', 'Orden eliminada correctamente.');
    }
}