<?php

namespace App\Http\Controllers;

use App\Models\Product;
use Illuminate\Http\Request;
use Illuminate\Validation\ValidationException;
use Illuminate\Support\Facades\Log;

class StoreController extends Controller
{
    /**
     * Muestra la página de inicio con productos.
     */
    public function indexWelcome()
    {
        $products = Product::all();
        Log::info('indexWelcome: Productos cargados', ['count' => $products->count(), 'products' => $products->toArray()]);
        return view('welcome', compact('products'));
    }

    /**
     * Muestra la página de la tienda con todos los productos.
     */
    public function index()
    {
        $products = Product::all();
        Log::info('index: Productos cargados', ['count' => $products->count(), 'products' => $products->toArray()]);
        return view('store.index', compact('products'));
    }

    /**
     * Añade un producto al carrito.
     */
    public function addToCart(Request $request)
    {
        try {
            // Validar los datos de entrada
            $request->validate([
                'product_id' => 'required|integer|exists:products,id',
                'quantity' => 'required|integer|min:1'
            ]);

            $cart = session()->get('cart', []);
            $product = Product::find($request->product_id);

            if (!$product) {
                return response()->json([
                    'success' => false,
                    'message' => 'Producto no encontrado'
                ], 404);
            }

            $productId = $product->id;
            $quantity = (int) $request->quantity;
            // Seleccionar precio: discount_price si is_discounted es 1 y discount_price no es null, sino price
            $price = ($product->is_discounted && !is_null($product->discount_price)) ? floatval($product->discount_price) : floatval($product->price);

            if (isset($cart[$productId])) {
                $cart[$productId]['quantity'] += $quantity;
                $cart[$productId]['price'] = $price; // Actualizar precio
            } else {
                $cart[$productId] = [
                    'name' => $product->name,
                    'price' => $price,
                    'quantity' => $quantity,
                    'image' => $product->image
                ];
            }

            session()->put('cart', $cart);
            session()->save(); // Forzar guardado de la sesión
            Log::info('addToCart: Sesión actualizada', ['cart' => $cart]);

            return response()->json([
                'success' => true,
                'product_name' => $product->name,
                'cart' => $cart
            ]);
        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => $e->errors()['product_id'][0] ?? $e->errors()['quantity'][0] ?? 'Datos inválidos'
            ], 422);
        } catch (\Exception $e) {
            Log::error('addToCart: Error', ['message' => $e->getMessage()]);
            return response()->json([
                'success' => false,
                'message' => 'Error al procesar la solicitud: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Elimina un producto del carrito.
     */
    public function removeFromCart(Request $request)
    {
        try {
            // Validar los datos de entrada
            $request->validate([
                'product_id' => 'required|integer'
            ]);

            $cart = session()->get('cart', []);
            $productId = (int) $request->product_id;
            Log::info('removeFromCart: Intentando eliminar', ['product_id' => $productId, 'cart' => $cart]);

            if (isset($cart[$productId])) {
                unset($cart[$productId]);
                session()->put('cart', $cart);
                session()->save(); // Forzar guardado de la sesión
                Log::info('removeFromCart: Producto eliminado', ['cart' => $cart]);

                return response()->json([
                    'success' => true,
                    'message' => 'Producto eliminado del carrito',
                    'cart' => $cart
                ]);
            }

            Log::warning('removeFromCart: Producto no encontrado en carrito', ['product_id' => $productId]);
            return response()->json([
                'success' => false,
                'message' => 'Producto no encontrado en el carrito',
                'cart' => $cart
            ], 404);
        } catch (ValidationException $e) {
            Log::error('removeFromCart: Validación fallida', ['errors' => $e->errors()]);
            return response()->json([
                'success' => false,
                'message' => $e->errors()['product_id'][0] ?? 'ID de producto inválido'
            ], 422);
        } catch (\Exception $e) {
            Log::error('removeFromCart: Error', ['message' => $e->getMessage()]);
            return response()->json([
                'success' => false,
                'message' => 'Error al eliminar el producto: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Obtiene el contenido actual del carrito.
     */
    public function getCart()
    {
        $cart = session()->get('cart', []);
        Log::info('getCart: Carrito obtenido', ['cart' => $cart]);
        return response()->json(['cart' => $cart]);
    }

    /**
     * Muestra la página de checkout.
     */
    public function checkout()
    {
        $cart = session()->get('cart', []);
        Log::info('checkout: Carrito en checkout', ['cart' => $cart]);
        if (empty($cart)) {
            return redirect()->route('store')->with('error', 'El carrito está vacío.');
        }
        $total = array_sum(array_map(fn($item) => $item['price'] * $item['quantity'], $cart));
        return view('store.checkout', compact('cart', 'total'));
    }

    /**
     * Muestra los detalles de un producto específico.
     */
    public function show($id)
    {
        $product = Product::find($id);
        if (!$product) {
            abort(404, 'Producto no encontrado');
        }
        return view('store.show', compact('product'));
    }
}